-- GlobalExpress Add-on Services Database Schema
-- Complete service management system

USE globalexpress_db;

-- ============================================
-- 1. SERVICE CATEGORIES TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS service_categories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(100) NOT NULL,
    name_az VARCHAR(100) NOT NULL,
    icon VARCHAR(50),
    description TEXT,
    description_az TEXT,
    target_audience ENUM('individual', 'business', 'both') DEFAULT 'both',
    sort_order INT DEFAULT 0,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_active (is_active),
    INDEX idx_audience (target_audience)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 2. SERVICES TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS services (
    id INT AUTO_INCREMENT PRIMARY KEY,
    category_id INT NOT NULL,
    service_code VARCHAR(50) UNIQUE NOT NULL,
    name VARCHAR(200) NOT NULL,
    name_az VARCHAR(200) NOT NULL,
    description TEXT,
    description_az TEXT,
    price DECIMAL(10,2) NOT NULL,
    price_type ENUM('fixed', 'per_kg', 'per_item', 'percentage', 'per_month') DEFAULT 'fixed',
    calculation_base VARCHAR(50), -- 'package_weight', 'order_value', 'item_count', etc.
    min_price DECIMAL(10,2) DEFAULT 0,
    max_price DECIMAL(10,2),
    is_recurring TINYINT(1) DEFAULT 0,
    recurring_period ENUM('daily', 'weekly', 'monthly', 'yearly'),
    target_audience ENUM('individual', 'business', 'both') DEFAULT 'both',
    requires_approval TINYINT(1) DEFAULT 0,
    estimated_duration VARCHAR(50), -- "24 hours", "3-5 days", etc.
    icon VARCHAR(50),
    image_url VARCHAR(255),
    terms_conditions TEXT,
    terms_conditions_az TEXT,
    is_featured TINYINT(1) DEFAULT 0,
    is_active TINYINT(1) DEFAULT 1,
    sort_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES service_categories(id) ON DELETE CASCADE,
    INDEX idx_code (service_code),
    INDEX idx_active (is_active),
    INDEX idx_featured (is_featured),
    INDEX idx_audience (target_audience)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 3. SERVICE PACKAGES TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS service_packages (
    id INT AUTO_INCREMENT PRIMARY KEY,
    package_code VARCHAR(50) UNIQUE NOT NULL,
    name VARCHAR(200) NOT NULL,
    name_az VARCHAR(200) NOT NULL,
    description TEXT,
    description_az TEXT,
    target_audience ENUM('individual', 'business') NOT NULL,
    price DECIMAL(10,2) NOT NULL,
    price_type ENUM('fixed', 'per_month', 'per_shipment') DEFAULT 'fixed',
    discount_percentage DECIMAL(5,2) DEFAULT 0,
    features JSON, -- Array of features
    is_popular TINYINT(1) DEFAULT 0,
    is_active TINYINT(1) DEFAULT 1,
    sort_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_code (package_code),
    INDEX idx_active (is_active),
    INDEX idx_audience (target_audience)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 4. PACKAGE SERVICES (Many-to-Many)
-- ============================================
CREATE TABLE IF NOT EXISTS package_services (
    id INT AUTO_INCREMENT PRIMARY KEY,
    package_id INT NOT NULL,
    service_id INT NOT NULL,
    quantity INT DEFAULT 1,
    is_unlimited TINYINT(1) DEFAULT 0,
    FOREIGN KEY (package_id) REFERENCES service_packages(id) ON DELETE CASCADE,
    FOREIGN KEY (service_id) REFERENCES services(id) ON DELETE CASCADE,
    UNIQUE KEY unique_package_service (package_id, service_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 5. USER SERVICE ORDERS TABLE
-- ============================================
CREATE TABLE IF NOT EXISTS user_service_orders (
    id INT AUTO_INCREMENT PRIMARY KEY,
    order_number VARCHAR(50) UNIQUE NOT NULL,
    user_id INT NOT NULL,
    package_id INT, -- If part of a package
    service_id INT, -- If individual service
    service_code VARCHAR(50),
    service_name VARCHAR(200),
    related_tracking_number VARCHAR(100), -- Related shipment
    quantity INT DEFAULT 1,
    unit_price DECIMAL(10,2) NOT NULL,
    total_price DECIMAL(10,2) NOT NULL,
    currency VARCHAR(3) DEFAULT 'AZN',
    status ENUM('pending', 'in_progress', 'completed', 'cancelled', 'refunded') DEFAULT 'pending',
    payment_status ENUM('pending', 'paid', 'failed', 'refunded') DEFAULT 'pending',
    payment_method VARCHAR(50),
    payment_date DATETIME,
    notes TEXT,
    customer_notes TEXT,
    admin_notes TEXT,
    assigned_to INT, -- Staff member ID
    started_at DATETIME,
    completed_at DATETIME,
    cancelled_at DATETIME,
    cancellation_reason TEXT,
    attachments JSON, -- Array of file paths
    metadata JSON, -- Additional data (addresses, preferences, etc.)
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (service_id) REFERENCES services(id) ON DELETE SET NULL,
    FOREIGN KEY (package_id) REFERENCES service_packages(id) ON DELETE SET NULL,
    INDEX idx_order_number (order_number),
    INDEX idx_user (user_id),
    INDEX idx_status (status),
    INDEX idx_payment_status (payment_status),
    INDEX idx_tracking (related_tracking_number),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 6. USER PACKAGE SUBSCRIPTIONS
-- ============================================
CREATE TABLE IF NOT EXISTS user_package_subscriptions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    package_id INT NOT NULL,
    subscription_number VARCHAR(50) UNIQUE NOT NULL,
    start_date DATE NOT NULL,
    end_date DATE NOT NULL,
    status ENUM('active', 'expired', 'cancelled') DEFAULT 'active',
    payment_status ENUM('pending', 'paid', 'failed') DEFAULT 'pending',
    auto_renew TINYINT(1) DEFAULT 0,
    services_used JSON, -- Track usage per service
    total_paid DECIMAL(10,2) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (package_id) REFERENCES service_packages(id) ON DELETE CASCADE,
    INDEX idx_user (user_id),
    INDEX idx_status (status),
    INDEX idx_dates (start_date, end_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 7. SERVICE ORDER ACTIVITIES/LOGS
-- ============================================
CREATE TABLE IF NOT EXISTS service_order_activities (
    id INT AUTO_INCREMENT PRIMARY KEY,
    order_id INT NOT NULL,
    user_id INT,
    activity_type VARCHAR(50) NOT NULL, -- 'created', 'updated', 'status_changed', 'payment', 'note_added', etc.
    description TEXT,
    old_value TEXT,
    new_value TEXT,
    performed_by INT, -- User or admin who performed action
    ip_address VARCHAR(45),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (order_id) REFERENCES user_service_orders(id) ON DELETE CASCADE,
    INDEX idx_order (order_id),
    INDEX idx_type (activity_type),
    INDEX idx_created (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 8. SERVICE RATINGS & REVIEWS
-- ============================================
CREATE TABLE IF NOT EXISTS service_reviews (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT NOT NULL,
    service_id INT NOT NULL,
    order_id INT NOT NULL,
    rating INT NOT NULL CHECK (rating >= 1 AND rating <= 5),
    review_title VARCHAR(200),
    review_text TEXT,
    pros TEXT,
    cons TEXT,
    would_recommend TINYINT(1) DEFAULT 1,
    is_verified TINYINT(1) DEFAULT 1, -- Verified purchase
    is_approved TINYINT(1) DEFAULT 0,
    admin_response TEXT,
    helpful_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (service_id) REFERENCES services(id) ON DELETE CASCADE,
    FOREIGN KEY (order_id) REFERENCES user_service_orders(id) ON DELETE CASCADE,
    INDEX idx_service (service_id),
    INDEX idx_user (user_id),
    INDEX idx_approved (is_approved),
    INDEX idx_rating (rating)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 9. SERVICE PRICING RULES (Dynamic Pricing)
-- ============================================
CREATE TABLE IF NOT EXISTS service_pricing_rules (
    id INT AUTO_INCREMENT PRIMARY KEY,
    service_id INT NOT NULL,
    rule_name VARCHAR(100) NOT NULL,
    condition_type ENUM('weight_range', 'value_range', 'quantity_range', 'user_type', 'subscription_tier'),
    condition_min DECIMAL(10,2),
    condition_max DECIMAL(10,2),
    condition_value VARCHAR(100),
    price_modifier_type ENUM('fixed_price', 'percentage_discount', 'fixed_discount'),
    price_modifier_value DECIMAL(10,2),
    priority INT DEFAULT 0,
    is_active TINYINT(1) DEFAULT 1,
    valid_from DATE,
    valid_to DATE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (service_id) REFERENCES services(id) ON DELETE CASCADE,
    INDEX idx_service (service_id),
    INDEX idx_active (is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 10. SERVICE PROMOTIONS
-- ============================================
CREATE TABLE IF NOT EXISTS service_promotions (
    id INT AUTO_INCREMENT PRIMARY KEY,
    promo_code VARCHAR(50) UNIQUE NOT NULL,
    name VARCHAR(200) NOT NULL,
    description TEXT,
    discount_type ENUM('percentage', 'fixed_amount', 'free_service'),
    discount_value DECIMAL(10,2),
    free_service_id INT,
    applicable_to ENUM('all_services', 'specific_services', 'specific_categories', 'packages'),
    applicable_ids JSON, -- Array of service/category IDs
    min_purchase_amount DECIMAL(10,2),
    max_discount_amount DECIMAL(10,2),
    usage_limit INT,
    usage_count INT DEFAULT 0,
    per_user_limit INT DEFAULT 1,
    valid_from DATETIME NOT NULL,
    valid_to DATETIME NOT NULL,
    is_active TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (free_service_id) REFERENCES services(id) ON DELETE SET NULL,
    INDEX idx_code (promo_code),
    INDEX idx_active (is_active),
    INDEX idx_dates (valid_from, valid_to)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- 11. PROMO CODE USAGE
-- ============================================
CREATE TABLE IF NOT EXISTS promo_code_usage (
    id INT AUTO_INCREMENT PRIMARY KEY,
    promo_id INT NOT NULL,
    user_id INT NOT NULL,
    order_id INT NOT NULL,
    discount_amount DECIMAL(10,2) NOT NULL,
    used_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (promo_id) REFERENCES service_promotions(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (order_id) REFERENCES user_service_orders(id) ON DELETE CASCADE,
    INDEX idx_promo (promo_id),
    INDEX idx_user (user_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- VIEWS FOR ANALYTICS
-- ============================================

-- Service performance view
CREATE OR REPLACE VIEW service_performance AS
SELECT 
    s.id,
    s.service_code,
    s.name,
    s.category_id,
    COUNT(uso.id) as total_orders,
    SUM(CASE WHEN uso.status = 'completed' THEN 1 ELSE 0 END) as completed_orders,
    SUM(CASE WHEN uso.status = 'cancelled' THEN 1 ELSE 0 END) as cancelled_orders,
    SUM(CASE WHEN uso.payment_status = 'paid' THEN uso.total_price ELSE 0 END) as total_revenue,
    AVG(CASE WHEN sr.rating IS NOT NULL THEN sr.rating ELSE 0 END) as avg_rating,
    COUNT(DISTINCT sr.id) as review_count
FROM services s
LEFT JOIN user_service_orders uso ON s.id = uso.service_id
LEFT JOIN service_reviews sr ON s.id = sr.service_id
GROUP BY s.id;

-- User service spending view
CREATE OR REPLACE VIEW user_service_spending AS
SELECT 
    u.id as user_id,
    u.full_name,
    u.email,
    COUNT(DISTINCT uso.id) as total_orders,
    SUM(CASE WHEN uso.payment_status = 'paid' THEN uso.total_price ELSE 0 END) as total_spent,
    MAX(uso.created_at) as last_order_date
FROM users u
LEFT JOIN user_service_orders uso ON u.id = uso.user_id
GROUP BY u.id;

-- ============================================
-- INDEXES FOR PERFORMANCE
-- ============================================

-- Add composite indexes for common queries
CREATE INDEX idx_orders_user_status ON user_service_orders(user_id, status);
CREATE INDEX idx_orders_date_status ON user_service_orders(created_at, status);
CREATE INDEX idx_services_active_audience ON services(is_active, target_audience);

-- ============================================
-- INITIAL DATA - Service Categories
-- ============================================

INSERT INTO service_categories (name, name_az, icon, description, description_az, target_audience, sort_order) VALUES
('Shopping Assistance', 'Alış-veriş Köməyi', '🛍️', 'Help finding and purchasing items', 'Məhsul tapmaq və almaqda kömək', 'both', 1),
('Quality Control', 'Keyfiyyət Nəzarəti', '🔍', 'Inspection and verification services', 'Yoxlama və təsdiqləmə xidmətləri', 'both', 2),
('Packaging Services', 'Qablaşdırma Xidmətləri', '📦', 'Enhanced packaging options', 'Təkmilləşdirilmiş qablaşdırma', 'both', 3),
('Storage & Handling', 'Saxlama və İdarəetmə', '🏪', 'Warehousing and special handling', 'Anbar və xüsusi idarəetmə', 'both', 4),
('Documentation', 'Sənədləşdirmə', '📄', 'Document preparation and management', 'Sənəd hazırlığı və idarəsi', 'both', 5),
('Business Services', 'Biznes Xidmətləri', '🏢', 'Services for commercial clients', 'Kommersiya müştəriləri üçün xidmətlər', 'business', 6),
('Delivery Services', 'Çatdırılma Xidmətləri', '🚚', 'Last-mile and special delivery', 'Son mil və xüsusi çatdırılma', 'both', 7),
('Premium Services', 'Premium Xidmətlər', '💎', 'VIP and concierge services', 'VIP və konsierj xidmətləri', 'both', 8);

-- ============================================
-- VERIFICATION QUERIES
-- ============================================

-- Verify all tables were created
SELECT 
    TABLE_NAME, 
    TABLE_ROWS,
    CREATE_TIME
FROM information_schema.TABLES 
WHERE TABLE_SCHEMA = 'globalexpress_db' 
AND TABLE_NAME LIKE 'service%'
ORDER BY TABLE_NAME;

-- Show table structure summary
SELECT 
    TABLE_NAME,
    COUNT(*) as column_count
FROM information_schema.COLUMNS
WHERE TABLE_SCHEMA = 'globalexpress_db'
AND TABLE_NAME LIKE 'service%'
GROUP BY TABLE_NAME;
